<?php
// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once 'dbconfig.php';
include_once 'includes/antisql.php';

// Verificar se a conexão foi estabelecida
if (!$conn) {
    echo json_encode(['success' => false, 'message' => 'Erro de conexão com banco de dados: ' . mysqli_connect_error()]);
    exit;
}

// Processar ações via POST - DEVE estar no início antes de qualquer saída HTML
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $acao = isset($_POST['acao']) ? $_POST['acao'] : '';
    $id_pedido = isset($_POST['id_pedido']) ? (int)$_POST['id_pedido'] : 0;
    $id_cliente = isset($_POST['id_cliente']) ? (int)$_POST['id_cliente'] : 0;
    $id_vendas = isset($_POST['id_vendas']) ? (int)$_POST['id_vendas'] : 0;
    
    if ($acao === 'inserir') {
        $produto = isset($_POST['produto']) ? $_POST['produto'] : '';
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : 0;
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        if (!empty($produto) && $valor > 0) {
            $sql = "INSERT INTO vendas (id_pedido, produto, valor, data2) VALUES (?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("isds", $id_pedido, $produto, $valor, $data);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Venda adicionada com sucesso!', 'id_vendas' => $conn->insert_id]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao adicionar venda: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Produto e valor são obrigatórios.']);
            exit;
        }
    } elseif ($acao === 'alterar' && $id_vendas > 0) {
        $produto = isset($_POST['produto']) ? $_POST['produto'] : '';
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : 0;
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        if (!empty($produto) && $valor > 0) {
            $sql = "UPDATE vendas SET produto = ?, valor = ?, data2 = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("sdsi", $produto, $valor, $data, $id_vendas);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Venda atualizada com sucesso!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao atualizar venda: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Produto e valor são obrigatórios.']);
            exit;
        }
    } elseif ($acao === 'excluir' && $id_vendas > 0) {
        $sql = "DELETE FROM vendas WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_vendas);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Venda excluída com sucesso!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao excluir venda: ' . $stmt->error]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    }
}

// Processar ação buscar_venda via GET
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['acao']) && $_GET['acao'] === 'buscar_venda') {
    header('Content-Type: application/json');
    
    $id_vendas = isset($_GET['id_vendas']) ? (int)$_GET['id_vendas'] : 0;
    
    if ($id_vendas > 0) {
        $sql = "SELECT * FROM vendas WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_vendas);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $venda = $result->fetch_assoc();
                
                if ($venda) {
                    $valor_formatado = number_format($venda['valor'], 2, ',', '.');
                    
                    echo json_encode([
                        'success' => true,
                        'data' => [
                            'produto' => $venda['produto'],
                            'valor_formatado' => $valor_formatado,
                            'data_venda' => $venda['data2']
                        ]
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Venda não encontrada.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao buscar venda.']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID da venda inválido.']);
    }
    exit;
}

// Receber parâmetros
$id_pedido = isset($_GET['id_pedido']) ? (int)$_GET['id_pedido'] : 0;
$id_cliente = isset($_GET['id_cliente']) ? (int)$_GET['id_cliente'] : 0;

// Buscar informações do pedido e cliente
$info_pedido = null;
if ($id_pedido > 0 && $id_cliente > 0) {
    $sql = "SELECT p.id as pedido_id, p.data, u.nome as cliente_nome 
            FROM pedidos p 
            LEFT JOIN usuarios u ON p.id_cliente = u.id 
            WHERE p.id = ? AND p.id_cliente = ?";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("ii", $id_pedido, $id_cliente);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $info_pedido = $result->fetch_assoc();
        }
    }
}

// Buscar vendas do pedido
$vendas = [];
if ($id_pedido > 0) {
    $sql = "SELECT * FROM vendas WHERE id_pedido = ? ORDER BY data2 DESC, id DESC";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("i", $id_pedido);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $vendas[] = $row;
            }
        }
    }
}

// Calcular total de vendas
$total_vendas = 0;
foreach ($vendas as $venda) {
    $total_vendas += $venda['valor'];
}

// Função para formatar data
function formatarData($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return 'Sem data';
    }
    
    // Se já está no formato dd/mm/yyyy
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        return $data;
    }
    
    // Se está no formato yyyy-mm-dd
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        $timestamp = strtotime($data);
        if ($timestamp && $timestamp > 0) {
            return date('d/m/Y', $timestamp);
        }
    }
    
    return $data;
}

// Função para converter data para input date
function converterDataParaInput($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return '';
    }
    
    // Se já está no formato yyyy-mm-dd
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        return $data;
    }
    
    // Se está no formato dd/mm/yyyy
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        $partes = explode('/', $data);
        return $partes[2] . '-' . $partes[1] . '-' . $partes[0];
    }
    
    // Tentar converter outros formatos
    $timestamp = strtotime($data);
    if ($timestamp && $timestamp > 0) {
        return date('Y-m-d', $timestamp);
    }
    
    return '';
}
?>

<!-- Informações do Pedido -->
<?php if ($info_pedido): ?>
<div class="row mb-3">
    <div class="col-12">
        <div class="alert alert-info">
            <strong>Pedido ID:</strong> <?php echo htmlspecialchars($info_pedido['pedido_id']); ?> | 
            <strong>Data:</strong> <?php echo formatarData($info_pedido['data']); ?> |
            <strong>Cliente:</strong> <?php echo htmlspecialchars($info_pedido['cliente_nome']); ?>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Mensagens de alerta -->
<div class="row mb-4">
    <div class="col-12">
        <div id="alertMessages"></div>
    </div>
</div>

<!-- Lista de Vendas -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Vendas do Pedido</h5>
                <div class="d-flex align-items-center gap-2">
                    <button type="button" class="btn btn-success btn-sm" onclick="adicionarNovaLinha()" title="Adicionar Nova Venda">
                        <i class="fas fa-plus"></i> Nova Venda
                    </button>
                    <span class="badge bg-primary fs-6">Total: R$ <?php echo number_format($total_vendas, 2, ',', '.'); ?></span>
                </div>
            </div>
            <div class="card-body">
                <?php if (empty($vendas)): ?>
                <div class="text-center text-muted py-4">
                    <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                    <p>Nenhuma venda encontrada para este pedido.</p>
                </div>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover" id="tabelaVendas">
                        <thead class="table-dark">
                            <tr>
                                <th>Nº</th>
                                <th>Produto</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th width="120">Ações</th>
                            </tr>
                        </thead>
                        <tfoot>
                            <tr>
                                <th>Nº</th>
                                <th>Produto</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th>Ações</th>
                            </tr>
                        </tfoot>
                        <tbody>
                            <?php foreach ($vendas as $venda): ?>
                            <tr data-id="<?php echo $venda['id']; ?>">
                                <td><?php echo $venda['id']; ?></td>
                                <td class="editable-cell" data-field="produto" data-original="<?php echo htmlspecialchars($venda['produto']); ?>">
                                    <span class="cell-display"><?php echo htmlspecialchars($venda['produto']); ?></span>
                                    <input type="text" class="cell-input form-control form-control-sm" value="<?php echo htmlspecialchars($venda['produto']); ?>" style="display: none;">
                                </td>
                                <td class="editable-cell" data-field="valor" data-original="<?php echo $venda['valor']; ?>">
                                    <span class="cell-display">R$ <?php echo number_format($venda['valor'], 2, ',', '.'); ?></span>
                                    <input type="text" class="cell-input form-control form-control-sm valor-input" value="<?php echo $venda['valor']; ?>" style="display: none;">
                                </td>
                                <td class="editable-cell" data-field="data2" data-original="<?php echo $venda['data2']; ?>">
                                    <span class="cell-display"><?php echo formatarData($venda['data2']); ?></span>
                                    <input type="date" class="cell-input form-control form-control-sm" value="<?php echo converterDataParaInput($venda['data2']); ?>" style="display: none;">
                                </td>
                                <td>
                                    <button type="button" class="btn btn-sm btn-success btn-save" onclick="salvarEdicaoInline(<?php echo $venda['id']; ?>)" title="Salvar" style="display: none;">
                                        <i class="fas fa-check"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="cancelarEdicaoInline(<?php echo $venda['id']; ?>)" title="Cancelar" style="display: none;">
                                        <i class="fas fa-times"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-warning btn-edit" onclick="editarInline(<?php echo $venda['id']; ?>)" title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" onclick="excluirVenda(<?php echo $venda['id']; ?>)" title="Excluir">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- CSS Específico do Modal (Reutilizando estilos do pedidos.php) -->
<!-- DataTables CSS -->
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap4.min.css">

<style>
/* Estilos mínimos específicos para edição inline */
.editable-cell { cursor: pointer; position: relative; }
.editable-cell:hover { background-color: #f8f9fa; }
.editable-cell.editing { background-color: #fff3cd; cursor: default; }
.cell-input { width: 100%; border: 2px solid #007bff; background-color: #fff; }
.cell-input:focus { border-color: #0056b3; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
.valor-input { text-align: right; }
.editing-row { background-color: #fff3cd !important; }
.nova-linha { background-color: #d1ecf1 !important; }

/* Estilos para os filtros de coluna */
.md-input-wrapper { margin-bottom: 0; }
.footer-search-input-vendas { width: 100%; padding: 4px 8px; font-size: 13px; border: 1px solid #ced4da; border-radius: 4px; }
.footer-search-input-vendas:focus { border-color: #80bdff; outline: 0; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
#tabelaVendas tfoot { background-color: #f8f9fa; }
#tabelaVendas tfoot th { padding: 8px 4px; font-weight: normal; }
</style>

<!-- JavaScript (Reutilizando bibliotecas já carregadas) -->
<!-- DataTables JS -->
<script type="text/javascript" src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script type="text/javascript" src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

<script>
// Variáveis globais
const ID_PEDIDO = <?php echo $id_pedido; ?>;
const ID_CLIENTE = <?php echo $id_cliente; ?>;

// Função para editar linha inline
function editarInline(idVenda) {
    cancelarTodasEdicoes();
    
    const linha = $(`tr[data-id="${idVenda}"]`);
    linha.addClass('editing-row');
    
    // Preparar o valor antes de mostrar o input
    const valorCell = linha.find('[data-field="valor"]');
    const valorOriginal = valorCell.data('original');
    const valorInput = valorCell.find('.valor-input');
    
    // Formatar o valor para exibição no input
    if (valorOriginal) {
        valorInput.val(parseFloat(valorOriginal).toLocaleString('pt-BR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        }));
    }
    
    linha.find('.cell-display').hide();
    linha.find('.cell-input').show();
    linha.find('.btn-edit, .btn-danger').hide();
    linha.find('.btn-save, .btn-cancel').show();
    
    // Aplicar máscara no campo valor
    aplicarMascaraValorInline(valorInput);
    
    linha.find('input').first().focus();
}

// Função para cancelar edição
function cancelarEdicaoInline(idVenda) {
    const linha = $(`tr[data-id="${idVenda}"]`);
    linha.removeClass('editing-row');
    
    // Restaurar valores originais
    linha.find('.editable-cell').each(function() {
        const campo = $(this).data('field');
        const valorOriginal = $(this).data('original');
        
        if (campo === 'valor') {
            // Usar o valor original sem formatação adicional
            $(this).find('.cell-input').val(valorOriginal);
        } else {
            $(this).find('.cell-input').val(valorOriginal);
        }
    });
    
    linha.find('.cell-input').hide();
    linha.find('.cell-display').show();
    linha.find('.btn-save, .btn-cancel').hide();
    linha.find('.btn-edit, .btn-danger').show();
}

// Função para salvar edição
function salvarEdicaoInline(idVenda) {
    const linha = $(`tr[data-id="${idVenda}"]`);
    
    const produto = linha.find('[data-field="produto"] .cell-input').val();
    const valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    const data = linha.find('[data-field="data2"] .cell-input').val();
    
    const valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    if (!produto.trim() || !valorParaBanco || isNaN(parseFloat(valorParaBanco)) || !data) {
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Todos os campos são obrigatórios e devem ser válidos.' });
        return;
    }
    
    fetch('vendas_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'alterar',
            id_vendas: idVenda,
            produto: produto,
            valor: valorParaBanco,
            data: data,
            id_pedido: ID_PEDIDO,
            id_cliente: ID_CLIENTE
        })
    })
    .then(response => response.json())
    .then(responseData => {
        if (responseData.success) {
            // Atualizar displays
            linha.find('[data-field="produto"] .cell-display').text(produto);
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2, maximumFractionDigits: 2
            }));
            
            // Usar a data original que foi enviada para o servidor
            if (data) {
                const dataFormatada = formatarDataParaExibicao(data);
                linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            }
            
            // Atualizar dados originais
            linha.find('[data-field="produto"]').data('original', produto);
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Sair do modo de edição
            linha.removeClass('editing-row');
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            linha.find('.btn-save, .btn-cancel').hide();
            linha.find('.btn-edit, .btn-danger').show();
            
            Swal.fire({ icon: 'success', title: 'Sucesso!', text: responseData.message, timer: 1500, showConfirmButton: false });
        } else {
            Swal.fire({ icon: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para formatar data para exibição
function formatarDataParaExibicao(data) {
    if (!data || data === '0000-00-00' || data === '1970-01-01') {
        return 'Sem data';
    }
    
    // Se a data já está no formato dd/mm/yyyy, retorna como está
    if (/^\d{2}\/\d{2}\/\d{4}$/.test(data)) {
        return data;
    }
    
    // Se a data está no formato yyyy-mm-dd, converte para dd/mm/yyyy
    if (/^\d{4}-\d{2}-\d{2}$/.test(data)) {
        try {
            const [ano, mes, dia] = data.split('-');
            return `${dia}/${mes}/${ano}`;
        } catch (e) {
            return data;
        }
    }
    
    // Tenta usar Date() como último recurso
    try {
        const dateObj = new Date(data + 'T00:00:00');
        if (!isNaN(dateObj.getTime())) {
            return dateObj.toLocaleDateString('pt-BR');
        }
    } catch (e) {
        // Se falhar, retorna a data original
    }
    
    return data;
}

// Função para cancelar todas as edições
function cancelarTodasEdicoes() {
    $('.editing-row').each(function() {
        const idVenda = $(this).data('id');
        if (typeof idVenda === 'string' && idVenda.startsWith('novo_')) {
            $(this).remove();
        } else {
            cancelarEdicaoInline(idVenda);
        }
    });
}

// Função para aplicar máscara de valor
function aplicarMascaraValorInline(campo) {
    campo.off('.mascaraInline');
    
    campo.on('input.mascaraInline', function() {
        let valor = this.value.replace(/\D/g, '');
        if (valor.length === 0) { this.value = ''; return; }
        
        // Se tem apenas 1 dígito, trata como centavos (0,0X)
        if (valor.length === 1) {
            this.value = '0,0' + valor;
            return;
        }
        
        // Se tem apenas 2 dígitos, trata como centavos (0,XX)
        if (valor.length === 2) {
            this.value = '0,' + valor;
            return;
        }
        
        // Para 3 ou mais dígitos, os últimos 2 são centavos
        const parteDecimal = valor.slice(-2);
        let parteInteira = valor.slice(0, -2);
        
        // Adicionar pontos para milhares apenas se necessário
        if (parteInteira.length > 3) {
            parteInteira = parteInteira.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        }
        
        this.value = parteInteira + ',' + parteDecimal;
    });
    
    campo.on('focus.mascaraInline', function() {
        setTimeout(() => this.select(), 10);
    });
}

// Função para adicionar nova linha
function adicionarNovaLinha() {
    cancelarTodasEdicoes();
    
    const novoId = 'novo_' + Date.now();
    const novaLinha = `
        <tr data-id="${novoId}" class="editing-row nova-linha">
            <td>Novo</td>
            <td class="editable-cell" data-field="produto" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="text" class="form-control form-control-sm cell-input" placeholder="Digite o produto" style="display: block;">
            </td>
            <td class="editable-cell" data-field="valor" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="text" class="form-control form-control-sm cell-input valor-input" placeholder="0,00" style="display: block;">
            </td>
            <td class="editable-cell" data-field="data2" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="date" class="form-control form-control-sm cell-input" style="display: block;">
            </td>
            <td>
                <button type="button" class="btn btn-success btn-sm btn-save" onclick="salvarNovaVenda('${novoId}')" title="Salvar">
                    <i class="fas fa-save"></i>
                </button>
                <button type="button" class="btn btn-secondary btn-sm btn-cancel ms-1" onclick="cancelarNovaVenda('${novoId}')" title="Cancelar">
                    <i class="fas fa-times"></i>
                </button>
            </td>
        </tr>
    `;
    
    $('#tabelaVendas tbody').append(novaLinha);
    
    const valorInput = $(`tr[data-id="${novoId}"] .valor-input`);
    aplicarMascaraValorInline(valorInput);
    
    $(`tr[data-id="${novoId}"] input`).first().focus();
}

// Função para salvar nova venda
function salvarNovaVenda(novoId) {
    const linha = $(`tr[data-id="${novoId}"]`);
    
    const produto = linha.find('[data-field="produto"] .cell-input').val();
    const valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    const data = linha.find('[data-field="data2"] .cell-input').val();
    
    const valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    if (!produto.trim() || !valorParaBanco || isNaN(parseFloat(valorParaBanco)) || !data) {
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Todos os campos são obrigatórios.' });
        return;
    }
    
    fetch('vendas_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'inserir',
            produto: produto,
            valor: valorParaBanco,
            data: data,
            id_pedido: ID_PEDIDO,
            id_cliente: ID_CLIENTE
        })
    })
    .then(response => response.json())
    .then(responseData => {
        if (responseData.success) {
            const novoIdVenda = responseData.id_vendas || 'novo';
            linha.attr('data-id', novoIdVenda);
            linha.removeClass('nova-linha');
            
            // Atualizar displays
            linha.find('[data-field="produto"] .cell-display').text(produto);
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2, maximumFractionDigits: 2
            }));
            
            // Usar a data original que foi enviada para o servidor
            if (data) {
                const dataFormatada = formatarDataParaExibicao(data);
                linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            }
            
            // Atualizar dados originais
            linha.find('[data-field="produto"]').data('original', produto);
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Sair do modo de edição
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            linha.removeClass('editing-row');
            
            // Atualizar botões
            linha.find('td:last').html(`
                <button type="button" class="btn btn-warning btn-sm btn-edit" onclick="editarInline(${novoIdVenda})" title="Editar">
                    <i class="fas fa-edit"></i>
                </button>
                <button type="button" class="btn btn-danger btn-sm ms-1" onclick="excluirVenda(${novoIdVenda})" title="Excluir">
                    <i class="fas fa-trash"></i>
                </button>
            `);
            
            Swal.fire({ icon: 'success', title: 'Sucesso!', text: 'Venda adicionada com sucesso!', timer: 1500, showConfirmButton: false });
        } else {
            Swal.fire({ icon: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para cancelar nova venda
function cancelarNovaVenda(novoId) {
    $(`tr[data-id="${novoId}"]`).remove();
}

// Função para excluir venda
function excluirVenda(idVendas) {
    Swal.fire({
        title: 'Tem certeza?',
        text: 'Esta ação não pode ser desfeita!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sim, excluir!',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch('vendas_modal.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    acao: 'excluir',
                    id_vendas: idVendas,
                    id_pedido: ID_PEDIDO,
                    id_cliente: ID_CLIENTE
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    $(`tr[data-id="${idVendas}"]`).remove();
                    Swal.fire({ icon: 'success', title: 'Excluído!', text: data.message, timer: 1500, showConfirmButton: false });
                } else {
                    Swal.fire({ icon: 'error', title: 'Erro!', text: data.message });
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
            });
        }
    });
}

// Inicialização
$(document).ready(function() {
    // Aplicar máscara nos campos de valor existentes
    $('.valor-input').each(function() {
        aplicarMascaraValorInline($(this));
    });
    
    // Adicionar evento de clique nos campos editáveis para edição direta
    $(document).on('click', '.editable-cell', function(e) {
        // Verificar se não está clicando em um input já aberto
        if ($(e.target).is('input') || $(e.target).closest('input').length > 0) {
            return;
        }
        
        // Verificar se a linha já está em modo de edição
        const linha = $(this).closest('tr');
        if (linha.hasClass('editing-row')) {
            return;
        }
        
        // Obter o ID da venda
        const idVenda = linha.data('id');
        
        // Verificar se é uma linha válida (não é nova linha)
        if (idVenda && !String(idVenda).startsWith('novo_')) {
            editarInline(idVenda);
        }
    });
    
    // Inicializar DataTable com filtros por coluna
    if ($.fn.DataTable && $('#tabelaVendas').length > 0) {
        // Verificar se a tabela já foi inicializada e destruir se necessário
        if ($.fn.DataTable.isDataTable('#tabelaVendas')) {
            $('#tabelaVendas').DataTable().destroy();
        }
        
        var table = $('#tabelaVendas').DataTable({
            "language": {
                "url": "datatable/pt-BR.json"
            },
            "pageLength": 10,
            "responsive": true,
            "processing": false,
            "info": true,
            "search": {
                "regex": false,
                "caseInsensitive": true
            },
            "initComplete": function () {
                console.log('🚀 DataTable Vendas inicializado');
                setupFooterSearchVendas();
            },
            "drawCallback": function (settings) {
                // Reinicializar tooltips nas novas linhas
                $('[data-toggle="tooltip"]').tooltip('dispose').tooltip({
                    delay: { show: 300, hide: 100 }
                });
                
                // Verificar se os inputs ainda existem após redraw
                var $inputs = $('.footer-search-input-vendas');
                if ($inputs.length === 0) {
                    console.log('⚠️ DRAW CALLBACK: Inputs perdidos após redraw, recriando...');
                    setTimeout(function() {
                        criarInputsVendasManualmente();
                    }, 200);
                }
            }
        });
        
        // Função para configurar busca no footer
        function setupFooterSearchVendas() {
            setTimeout(function() {
                console.log('🔧 SETUP FOOTER SEARCH VENDAS - Iniciando...');
                
                // Verificar se já existem inputs criados
                var $inputsExistentes = $('.footer-search-input-vendas');
                if ($inputsExistentes.length > 0) {
                    console.log('✅ Inputs já criados, pulando criação');
                    return;
                }
                
                // Tentar método automático primeiro
                var inputsCriados = 0;
                try {
                    table.columns().every(function (index) {
                        var column = this;
                        var title = $(column.footer()).text().trim();
                        
                        // Pular coluna de Ações (última coluna)
                        if (index === 4) {
                            $(column.footer()).html('');
                            return;
                        }
                        
                        if (title) {
                            try {
                                // Criar input com wrapper
                                var inputHtml = '<div class="md-input-wrapper"><input type="text" class="md-form-control footer-search-input-vendas" placeholder="' + title.replace(/'/g, '&#39;').replace(/"/g, '&quot;') + '" data-column="' + index + '" /></div>';
                                $(column.footer()).html(inputHtml);
                                inputsCriados++;
                                
                                // Configurar evento de busca com throttling
                                var searchTimeout;
                                $('input', column.footer()).on('keyup change clear', function () {
                                    var searchValue = this.value;
                                    
                                    // Limpar timeout anterior
                                    clearTimeout(searchTimeout);
                                    
                                    searchTimeout = setTimeout(function() {
                                        if (column.search() !== searchValue) {
                                            column.search(searchValue, false, false).draw();
                                        }
                                    }, 300);
                                });
                                
                                // Prevenir ordenação ao clicar no input
                                $('input', column.footer()).on('click', function(e) {
                                    e.stopPropagation();
                                });
                                
                            } catch (error) {
                                console.log('❌ Erro ao criar input para coluna', index, ':', error);
                            }
                        }
                    });
                } catch (error) {
                    console.log('❌ Erro no método automático:', error);
                    inputsCriados = 0;
                }
                
                // Se não conseguiu criar automaticamente, forçar criação manual
                if (inputsCriados === 0) {
                    console.log('⚠️ Método automático falhou, forçando criação manual...');
                    criarInputsVendasManualmente();
                } else {
                    console.log('✅ Setup concluído com sucesso! Inputs criados:', inputsCriados);
                }
            }, 300);
        }
        
        // Função para criar inputs manualmente
        function criarInputsVendasManualmente() {
            console.log('🔧 CRIAÇÃO MANUAL VENDAS - Iniciando...');
            
            // Definir cabeçalhos das colunas
            var colunas = [
                { titulo: 'Nº', placeholder: 'Nº' },
                { titulo: 'Produto', placeholder: 'Produto' },
                { titulo: 'Valor', placeholder: 'Valor' },
                { titulo: 'Data', placeholder: 'Data' },
                { titulo: 'Ações', placeholder: '' }
            ];
            
            // Verificar se footer existe
            var $footer = $('#tabelaVendas tfoot');
            if ($footer.length === 0) {
                console.log('❌ Footer não encontrado, criando...');
                $('#tabelaVendas').append('<tfoot><tr>' + '<th></th>'.repeat(5) + '</tr></tfoot>');
                $footer = $('#tabelaVendas tfoot');
            }
            
            var $footerCells = $footer.find('th');
            
            // Criar inputs para cada coluna (exceto a última - Ações)
            for (var i = 0; i < colunas.length - 1; i++) {
                var coluna = colunas[i];
                var $cell = $footerCells.eq(i);
                
                if ($cell.length > 0) {
                    // Criar input com wrapper
                    var inputHtml = '<div class="md-input-wrapper">' +
                        '<input type="text" class="md-form-control footer-search-input-vendas" ' +
                        'placeholder="' + coluna.placeholder + '" data-column="' + i + '" />' +
                        '</div>';
                    
                    $cell.html(inputHtml);
                    
                    // Configurar evento de busca com throttling
                    var $input = $cell.find('.footer-search-input-vendas');
                    (function(columnIndex) {
                        $input.on('keyup change clear', function() {
                            var searchValue = this.value;
                            
                            // Aplicar filtro na coluna com throttling
                            clearTimeout(window['timeout_vendas_col_' + columnIndex]);
                            window['timeout_vendas_col_' + columnIndex] = setTimeout(function() {
                                if (table.column(columnIndex).search() !== searchValue) {
                                    table.column(columnIndex).search(searchValue, false, false).draw();
                                }
                            }, 300);
                        });
                        
                        // Prevenir ordenação ao clicar no input
                        $input.on('click', function(e) {
                            e.stopPropagation();
                        });
                    })(i);
                }
            }
            
            // Limpar última coluna (Ações)
            if ($footerCells.length > 4) {
                $footerCells.eq(4).html('');
            }
            
            console.log('🔧 CRIAÇÃO MANUAL VENDAS - Concluída!');
        }
    }
});
</script>